<?php

namespace App\Livewire\Admin;

use App\Services\FlashMessageService;
use Livewire\Component;
use Illuminate\Support\Facades\File;

class EmailSettings extends Component
{
    public $smtpSettings = [
        'MAIL_MAILER' => '',
        'MAIL_HOST' => '',
        'MAIL_PORT' => '',
        'MAIL_USERNAME' => '',
        'MAIL_PASSWORD' => '',
        'MAIL_ENCRYPTION' => '',
        'MAIL_FROM_ADDRESS' => '',
        'MAIL_FROM_NAME' => '',
    ];

    public $testEmail = '';
    public $testEmailSubject = 'Test Email from LLN Project';
    public $testEmailMessage = 'This is a test email to verify that email notifications are working correctly.';

    public function mount()
    {
        $this->loadSmtpSettings();
    }

    public function loadSmtpSettings()
    {
        // Load from .env file
        $envPath = base_path('.env');
        if (File::exists($envPath)) {
            $envContent = File::get($envPath);
            $lines = explode("\n", $envContent);
            
            foreach ($lines as $lineNum => $line) {
                $originalLine = $line;
                $line = trim($line);
                
                // Skip empty lines and comments
                if (empty($line) || strpos($line, '#') === 0) {
                    continue;
                }
                
                // Check if line contains =
                if (strpos($line, '=') !== false) {
                    $parts = explode('=', $line, 2);
                    $key = trim($parts[0]);
                    $value = isset($parts[1]) ? trim($parts[1]) : '';
                    
                    // Remove quotes if they exist
                    $value = trim($value, '"\'');
                    
                    // Update our settings array if this key exists
                    if (isset($this->smtpSettings[$key])) {
                        $this->smtpSettings[$key] = $value;
                    }
                }
            }
        }
        
        // Set default values if not found in .env
        if (empty($this->smtpSettings['MAIL_MAILER'])) {
            $this->smtpSettings['MAIL_MAILER'] = 'smtp';
        }
        if (empty($this->smtpSettings['MAIL_HOST'])) {
            $this->smtpSettings['MAIL_HOST'] = '';
        }
        if (empty($this->smtpSettings['MAIL_PORT'])) {
            $this->smtpSettings['MAIL_PORT'] = '';
        }
        if (empty($this->smtpSettings['MAIL_USERNAME'])) {
            $this->smtpSettings['MAIL_USERNAME'] = '';
        }
        if (empty($this->smtpSettings['MAIL_PASSWORD'])) {
            $this->smtpSettings['MAIL_PASSWORD'] = '';
        }
        if (empty($this->smtpSettings['MAIL_ENCRYPTION'])) {
            $this->smtpSettings['MAIL_ENCRYPTION'] = '';
        }
        if (empty($this->smtpSettings['MAIL_FROM_ADDRESS'])) {
            $this->smtpSettings['MAIL_FROM_ADDRESS'] = '';
        }
        if (empty($this->smtpSettings['MAIL_FROM_NAME'])) {
            $this->smtpSettings['MAIL_FROM_NAME'] = '';
        }
        
    }

    public function refreshSettings()
    {
        $this->loadSmtpSettings();
        FlashMessageService::success('Settings refreshed from .env file!', 'Settings Refreshed');
        $this->dispatch('flashMessage', 'success', 'Settings refreshed from .env file!', 'Settings Refreshed', 3000);
    }


    public function updateSmtpSettings()
    {
        $this->validate([
            'smtpSettings.MAIL_MAILER' => 'required|string',
            'smtpSettings.MAIL_HOST' => 'required|string',
            'smtpSettings.MAIL_PORT' => 'required|numeric',
            'smtpSettings.MAIL_USERNAME' => 'required|string',
            'smtpSettings.MAIL_PASSWORD' => 'required|string',
            'smtpSettings.MAIL_ENCRYPTION' => 'required|string',
            'smtpSettings.MAIL_FROM_ADDRESS' => 'required|email',
            'smtpSettings.MAIL_FROM_NAME' => 'required|string',
        ]);

        try {
            $this->updateEnvFile();
            
            FlashMessageService::success('SMTP settings updated successfully!', 'SMTP Updated');
            $this->dispatch('flashMessage', 'success', 'SMTP settings updated successfully!', 'SMTP Updated', 5000);

        } catch (\Exception $e) {
            FlashMessageService::error('Failed to update SMTP settings: ' . $e->getMessage(), 'SMTP Error');
            $this->dispatch('flashMessage', 'error', 'Failed to update SMTP settings: ' . $e->getMessage(), 'SMTP Error', 8000);
            \Log::error('SMTP settings update error: ' . $e->getMessage());
        }
    }

    private function updateEnvFile()
    {
        $envPath = base_path('.env');
        $envContent = File::get($envPath);
        $lines = explode("\n", $envContent);
        
        $updatedLines = [];
        $updatedKeys = [];
        
        foreach ($lines as $line) {
            if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
                list($key, $value) = explode('=', $line, 2);
                $key = trim($key);
                
                if (isset($this->smtpSettings[$key])) {
                    // Add quotes around the value if it contains spaces or special characters
                    $newValue = $this->smtpSettings[$key];
                    if (strpos($newValue, ' ') !== false || strpos($newValue, '#') !== false) {
                        $newValue = '"' . $newValue . '"';
                    }
                    $updatedLines[] = $key . '=' . $newValue;
                    $updatedKeys[] = $key;
                } else {
                    $updatedLines[] = $line;
                }
            } else {
                $updatedLines[] = $line;
            }
        }
        
        // Add any missing keys
        foreach ($this->smtpSettings as $key => $value) {
            if (!in_array($key, $updatedKeys)) {
                // Add quotes around the value if it contains spaces or special characters
                $newValue = $value;
                if (strpos($newValue, ' ') !== false || strpos($newValue, '#') !== false) {
                    $newValue = '"' . $newValue . '"';
                }
                $updatedLines[] = $key . '=' . $newValue;
            }
        }
        
        File::put($envPath, implode("\n", $updatedLines));
    }

    public function sendTestEmail()
    {
        $this->validate([
            'testEmail' => 'required|email',
            'testEmailSubject' => 'required|string|max:255',
            'testEmailMessage' => 'required|string|max:1000',
        ]);

        try {
            // Create a simple test email
            $emailData = [
                'subject' => $this->testEmailSubject,
                'message' => $this->testEmailMessage,
                'siteName' => config('app.name'),
                'timestamp' => now()->format('Y-m-d H:i:s'),
            ];

            // Use the EmailNotificationService to send the test email
            $emailSent = \App\Services\EmailNotificationService::sendEmailIfEnabled(
                new \App\Mail\TestEmail($emailData),
                $this->testEmail,
                $this->testEmailSubject
            );

            if ($emailSent) {
                FlashMessageService::success('Test email sent successfully!', 'Email Sent');
                $this->dispatch('flashMessage', 'success', 'Test email sent successfully!', 'Email Sent', 5000);
            } else {
                FlashMessageService::warning('Email notifications are disabled. Test email was not sent.', 'Email Disabled');
                $this->dispatch('flashMessage', 'warning', 'Email notifications are disabled. Test email was not sent.', 'Email Disabled', 8000);
            }

        } catch (\Exception $e) {
            FlashMessageService::error('Failed to send test email: ' . $e->getMessage(), 'Email Error');
            $this->dispatch('flashMessage', 'error', 'Failed to send test email: ' . $e->getMessage(), 'Email Error', 8000);
            \Log::error('Test email error: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.email-settings');
    }
}