<?php

namespace App\Livewire\Admin;

use App\Models\EmailTemplate;
use App\Services\FlashMessageService;
use Livewire\Component;
use Illuminate\Support\Str;

class EmailTemplateForm extends Component
{
    public $templateId = null;
    public $name = '';
    public $slug = '';
    public $subject = '';
    public $body = '';
    public $type = 'general';
    public $is_active = true;
    public $description = '';
    public $variables = [];

    public $editMode = false;
    public $availableVariables = [];
    public $selectedVariables = [];

    protected $rules = [
        'name' => 'required|string|max:255',
        'subject' => 'required|string|max:500',
        'body' => 'required|string',
        'type' => 'required|string',
        'is_active' => 'boolean',
        'description' => 'nullable|string|max:1000',
    ];

    public function mount($templateId = null)
    {
        $this->availableVariables = EmailTemplate::getAvailableVariables();
        
        if ($templateId) {
            $this->editMode = true;
            $this->templateId = $templateId;
            $template = EmailTemplate::findOrFail($templateId);
            
            $this->name = $template->name;
            $this->slug = $template->slug;
            $this->subject = $template->subject;
            $this->body = $template->body;
            $this->type = $template->type;
            $this->is_active = $template->is_active;
            $this->description = $template->description;
            $this->variables = $template->variables ?? [];
            $this->selectedVariables = $template->variables ?? [];
        } else {
            $this->generateSlug();
        }
    }

    public function updatedName()
    {
        $this->generateSlug();
    }

    public function generateSlug()
    {
        $this->slug = Str::slug($this->name);
    }

    public function removeVariable($index)
    {
        unset($this->selectedVariables[$index]);
        $this->variables = array_values($this->selectedVariables);
    }

    public function save()
    {
        // Check permissions
        if ($this->editMode && !auth()->user()->can('email-templates.edit')) {
            FlashMessageService::insufficientPermissions();
            $this->dispatch('flashMessage', 'error', 'You do not have permission to edit email templates.', 'Insufficient Permissions', 8000);
            return;
        } elseif (!$this->editMode && !auth()->user()->can('email-templates.create')) {
            FlashMessageService::insufficientPermissions();
            $this->dispatch('flashMessage', 'error', 'You do not have permission to create email templates.', 'Insufficient Permissions', 8000);
            return;
        }

        if ($this->editMode) {
            $this->rules['name'] = 'required|string|max:255|unique:email_templates,name,' . $this->templateId;
        } else {
            $this->rules['name'] = 'required|string|max:255|unique:email_templates,name';
        }

        $this->validate();

        $data = [
            'name' => $this->name,
            'slug' => $this->slug,
            'subject' => $this->subject,
            'body' => $this->body,
            'type' => $this->type,
            'is_active' => $this->is_active,
            'description' => $this->description,
            'variables' => $this->selectedVariables,
        ];

        try {
            if ($this->editMode) {
                $template = EmailTemplate::findOrFail($this->templateId);
                $template->update($data);
                
                FlashMessageService::success('Email template updated successfully!', 'Template Updated');
                $this->dispatch('flashMessage', 'success', 'Email template updated successfully!', 'Template Updated', 5000);
            } else {
                EmailTemplate::create($data);
                
                FlashMessageService::success('Email template created successfully!', 'Template Created');
                $this->dispatch('flashMessage', 'success', 'Email template created successfully!', 'Template Created', 5000);
            }

            return redirect()->route('admin.email-templates');
            
        } catch (\Exception $e) {
            FlashMessageService::error('Failed to save email template. Please try again.', 'Save Failed');
            $this->dispatch('flashMessage', 'error', 'Failed to save email template. Please try again.', 'Save Failed', 8000);
        }
    }

    public function cancel()
    {
        return redirect()->route('admin.email-templates');
    }

    public function render()
    {
        return view('livewire.admin.email-template-form', [
            'typeOptions' => EmailTemplate::getTypeOptions(),
        ]);
    }
}
