<?php

namespace App\Livewire\Admin;

use App\Models\EmailTemplate;
use App\Services\FlashMessageService;
use Livewire\Component;
use Livewire\WithPagination;

class EmailTemplateManagement extends Component
{
    use WithPagination;

    public $search = '';
    public $typeFilter = '';
    public $statusFilter = '';
    public $sortField = 'created_at';
    public $sortDirection = 'desc';
    
    // Preview modal properties
    public $showPreviewModal = false;
    public $previewTemplateData = null;
    public $previewSubject = '';
    public $previewBody = '';
    public $templateTypeStatus = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'typeFilter' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'sortField' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'desc'],
    ];

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        
        $this->sortField = $field;
    }

    public function clearFilters()
    {
        $this->search = '';
        $this->typeFilter = '';
        $this->statusFilter = '';
        $this->resetPage();
    }

    public function toggleStatus($templateId)
    {
        $template = EmailTemplate::findOrFail($templateId);
        $template->update(['is_active' => !$template->is_active]);
        
        FlashMessageService::success('Template status updated successfully!', 'Status Updated');
        $this->dispatch('flashMessage', 'success', 'Template status updated successfully!', 'Status Updated', 5000);
    }

    public function deleteTemplate($templateId)
    {
        $template = EmailTemplate::findOrFail($templateId);
        $template->delete();
        
        FlashMessageService::success('Template deleted successfully!', 'Template Deleted');
        $this->dispatch('flashMessage', 'success', 'Template deleted successfully!', 'Template Deleted', 5000);
    }

    public function previewTemplate($templateId)
    {
        $this->previewTemplateData = EmailTemplate::findOrFail($templateId);
        
        // Check if template type is active
        $this->templateTypeStatus = \App\Services\EmailTemplateService::getTemplateTypeStatus($this->previewTemplateData->slug);
        
        // Create sample data for preview
        $siteLogo = \App\Models\Setting::get('site_logo', '');
        $logoUrl = '';
        
        if ($siteLogo && \Storage::disk('public')->exists($siteLogo)) {
            $logoUrl = asset('storage/' . $siteLogo);
        }
        
        $sampleData = [
            'company_name' => 'LLN Project',
            'company_email' => 'admin@llnproject.com',
            'company_phone' => '+1-555-0123',
            'company_address' => '123 Learning Street, Education City, EC 12345',
            'learner_first_name' => 'John',
            'learner_last_name' => 'Doe',
            'learner_full_name' => 'John Doe',
            'learner_email' => 'john.doe@example.com',
            'learner_phone' => '+1-555-0456',
            'admin_first_name' => 'Admin',
            'admin_last_name' => 'User',
            'admin_full_name' => 'Admin User',
            'admin_email' => 'admin@llnproject.com',
            'site_name' => 'LLN Project',
            'site_url' => config('app.url'),
            'current_date' => now()->format('Y-m-d'),
            'current_time' => now()->format('H:i:s'),
            'year' => now()->year,
            'enrollment_date' => now()->format('Y-m-d'),
            'course_name' => 'LLN Assessment Course',
            'course_duration' => '6 weeks',
            'completion_date' => now()->addWeeks(6)->format('Y-m-d'),
            'company_logo' => $logoUrl,
            'site_logo' => $logoUrl,
            'logo_url' => $logoUrl,
        ];
        
        // Process the template with sample data
        $processedTemplate = $this->previewTemplateData->processTemplate($sampleData);
        
        $this->previewSubject = $processedTemplate['subject'];
        $this->previewBody = $processedTemplate['body'];
        $this->showPreviewModal = true;
    }

    public function closePreviewModal()
    {
        $this->showPreviewModal = false;
        $this->previewTemplateData = null;
        $this->previewSubject = '';
        $this->previewBody = '';
        $this->templateTypeStatus = [];
    }

    public function render()
    {
        $query = EmailTemplate::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('subject', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->typeFilter, function ($query) {
                $query->where('type', $this->typeFilter);
            })
            ->when($this->statusFilter !== '', function ($query) {
                $query->where('is_active', $this->statusFilter);
            })
            ->orderBy($this->sortField, $this->sortDirection);

        $templates = $query->paginate(10);

        return view('livewire.admin.email-template-management', [
            'templates' => $templates,
            'typeOptions' => EmailTemplate::getTypeOptions(),
        ]);
    }
}
