<?php

namespace App\Livewire\Admin;

use App\Models\EmailTemplateType;
use App\Services\FlashMessageService;
use Livewire\Component;
use Illuminate\Support\Str;

class EmailTemplateTypeForm extends Component
{
    public $typeId = null;
    public $name = '';
    public $slug = '';
    public $description = '';
    public $color = '#007bff';
    public $icon = 'fas fa-envelope';
    public $is_active = true;
    public $sort_order = 0;

    public $editMode = false;

    protected $rules = [
        'name' => 'required|string|max:255',
        'slug' => 'required|string|max:255',
        'description' => 'nullable|string|max:1000',
        'color' => 'required|string|max:7',
        'icon' => 'nullable|string|max:255',
        'is_active' => 'boolean',
        'sort_order' => 'required|integer|min:0',
    ];

    public function mount($typeId = null)
    {
        if ($typeId) {
            $this->editMode = true;
            $this->typeId = $typeId;
            $type = EmailTemplateType::findOrFail($typeId);
            
            $this->name = $type->name;
            $this->slug = $type->slug;
            $this->description = $type->description;
            $this->color = $type->color;
            $this->icon = $type->icon;
            $this->is_active = $type->is_active;
            $this->sort_order = $type->sort_order;
        } else {
            $this->generateSlug();
        }
    }

    public function updatedName()
    {
        $this->generateSlug();
    }

    public function generateSlug()
    {
        $this->slug = Str::slug($this->name);
    }

    public function save()
    {
        // Check permissions
        if ($this->editMode && !auth()->user()->can('email-template-types.edit')) {
            FlashMessageService::insufficientPermissions();
            $this->dispatch('flashMessage', 'error', 'You do not have permission to edit email template types.', 'Insufficient Permissions', 8000);
            return;
        } elseif (!$this->editMode && !auth()->user()->can('email-template-types.create')) {
            FlashMessageService::insufficientPermissions();
            $this->dispatch('flashMessage', 'error', 'You do not have permission to create email template types.', 'Insufficient Permissions', 8000);
            return;
        }

        if ($this->editMode) {
            $this->rules['name'] = 'required|string|max:255|unique:email_template_types,name,' . $this->typeId;
            $this->rules['slug'] = 'required|string|max:255|unique:email_template_types,slug,' . $this->typeId;
        } else {
            $this->rules['name'] = 'required|string|max:255|unique:email_template_types,name';
            $this->rules['slug'] = 'required|string|max:255|unique:email_template_types,slug';
        }

        $this->validate();

        $data = [
            'name' => $this->name,
            'slug' => $this->slug,
            'description' => $this->description,
            'color' => $this->color,
            'icon' => $this->icon,
            'is_active' => $this->is_active,
            'sort_order' => $this->sort_order,
        ];

        try {
            if ($this->editMode) {
                $type = EmailTemplateType::findOrFail($this->typeId);
                $type->update($data);
                
                FlashMessageService::success('Email template type updated successfully!', 'Type Updated');
                $this->dispatch('flashMessage', 'success', 'Email template type updated successfully!', 'Type Updated', 5000);
            } else {
                EmailTemplateType::create($data);
                
                FlashMessageService::success('Email template type created successfully!', 'Type Created');
                $this->dispatch('flashMessage', 'success', 'Email template type created successfully!', 'Type Created', 5000);
            }

            return redirect()->route('admin.email-template-types');
            
        } catch (\Exception $e) {
            FlashMessageService::error('Failed to save email template type. Please try again.', 'Save Failed');
            $this->dispatch('flashMessage', 'error', 'Failed to save email template type. Please try again.', 'Save Failed', 8000);
        }
    }

    public function cancel()
    {
        return redirect()->route('admin.email-template-types');
    }

    public function render()
    {
        return view('livewire.admin.email-template-type-form');
    }
}