<?php

namespace App\Livewire\Admin;

use App\Models\Quiz;
use App\Services\QuizService;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class QuizForm extends Component
{
    public $quiz;
    public $quizId;
    public $title;
    public $description;
    public $slug;
    public $settings;
    public $metadata;
    public $i18n;
    public $visibility;
    public $isActive;
    public $expiresAt;
    public $editMode = false;

    protected $rules = [
        'title' => 'required|string|max:255',
        'description' => 'nullable|string',
        'visibility' => 'required|in:private,public,unlisted',
        'isActive' => 'boolean',
        'expiresAt' => 'nullable|date|after:now',
    ];

    public function mount($quiz = null)
    {
        $this->quiz = $quiz;
        $this->editMode = $quiz !== null;

        if ($this->editMode) {
            $this->quizId = $quiz->id;
            $this->title = $quiz->title;
            $this->description = $quiz->description;
            $this->slug = $quiz->slug;
            $this->settings = $quiz->settings ?? [];
            $this->metadata = $quiz->metadata ?? [];
            $this->i18n = $quiz->i18n ?? [];
            $this->visibility = $quiz->visibility;
            $this->isActive = $quiz->is_active;
            $this->expiresAt = $quiz->expires_at?->format('Y-m-d\TH:i');
        } else {
            $this->visibility = 'private';
            $this->isActive = true;
            $this->settings = [
                'duration_minutes' => null,
                'max_attempts' => null,
                'show_correct_answers' => true,
                'randomize_questions' => false,
                'randomize_answers' => false,
                'allow_navigation' => true,
                'time_limit_per_question' => null,
            ];
            $this->metadata = [];
            $this->i18n = [];
        }
    }

    public function updatedTitle($value)
    {
        if (!$this->editMode) {
            $this->slug = Str::slug($value);
        }
    }

    public function save()
    {
        $this->validate();

        try {
            $quizService = app(QuizService::class);
            
            $data = [
                'title' => $this->title,
                'description' => $this->description,
                'slug' => $this->slug,
                'settings' => $this->settings,
                'metadata' => $this->metadata,
                'i18n' => $this->i18n,
                'visibility' => $this->visibility,
                'is_active' => $this->isActive,
                'expires_at' => $this->expiresAt ? \Carbon\Carbon::parse($this->expiresAt) : null,
            ];

            if ($this->editMode) {
                $quiz = Quiz::findOrFail($this->quizId);
                $quiz = $quizService->updateQuiz($quiz, $data, Auth::id());
                session()->flash('success', 'Quiz updated successfully.');
            } else {
                $quiz = $quizService->createQuiz($data, Auth::id());
                session()->flash('success', 'Quiz created successfully.');
            }

            return redirect()->route('admin.quizzes.show', $quiz);
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to save quiz: ' . $e->getMessage());
        }
    }

    public function cancel()
    {
        return redirect()->route('admin.quizzes');
    }

    public function render()
    {
        return view('livewire.admin.quiz-form');
    }
}