<?php

namespace App\Livewire\Admin;

use App\Models\Resource;
use App\Services\FlashMessageService;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;

class ResourceForm extends Component
{
    use WithFileUploads;

    public $title = '';
    public $description = '';
    public $file;
    public $is_active = true;

    public $editMode = false;
    public $resourceId = null;
    
    // Existing file information for edit mode
    public $existingFileName = '';
    public $existingFileSize = '';
    public $existingFileType = '';
    public $existingFileUrl = '';

    protected $rules = [
        'title' => 'required|string|max:255',
        'description' => 'nullable|string|max:1000',
        'file' => 'required|file|mimes:pdf,doc,docx|max:10240', // Max 10MB
        'is_active' => 'boolean',
    ];

    protected $messages = [
        'title.required' => 'Title is required.',
        'title.max' => 'Title must not exceed 255 characters.',
        'description.max' => 'Description must not exceed 1000 characters.',
        'file.required' => 'File is required.',
        'file.mimes' => 'File must be a PDF, DOC, or DOCX file.',
        'file.max' => 'File size must not exceed 10MB.',
    ];

    public function mount($resourceId = null)
    {
        if ($resourceId) {
            $this->editMode = true;
            $this->resourceId = $resourceId;
            $resource = Resource::findOrFail($resourceId);
            
            $this->title = $resource->title;
            $this->description = $resource->description;
            $this->is_active = $resource->is_active;
            
            // Load existing file information
            $this->existingFileName = $resource->file_name;
            $this->existingFileSize = $resource->formatted_file_size;
            $this->existingFileType = $resource->file_type;
            $this->existingFileUrl = $resource->file_url;
            
            // Make file optional in edit mode
            $this->rules['file'] = 'nullable|file|mimes:pdf,doc,docx|max:10240';
        }
    }

    public function updated($propertyName)
    {
        if ($this->editMode && $propertyName === 'file') {
            // For edit mode, make file optional
            $this->rules['file'] = 'nullable|file|mimes:pdf,doc,docx|max:10240';
        }

        $this->validateOnly($propertyName);
    }

    public function save()
    {
        // Check permissions
        if ($this->editMode && !auth()->user()->can('resources.edit')) {
            FlashMessageService::insufficientPermissions();
            $this->dispatch('flashMessage', 'error', 'You do not have permission to edit resources.', 'Insufficient Permissions', 8000);
            return;
        } elseif (!$this->editMode && !auth()->user()->can('resources.create')) {
            FlashMessageService::insufficientPermissions();
            $this->dispatch('flashMessage', 'error', 'You do not have permission to create resources.', 'Insufficient Permissions', 8000);
            return;
        }

        if ($this->editMode) {
            $this->rules['file'] = 'nullable|file|mimes:pdf,doc,docx|max:10240';
        }

        $this->validate();

        $resourceData = [
            'title' => $this->title,
            'description' => $this->description ?: null,
            'is_active' => $this->is_active,
            'created_by' => auth()->id(),
        ];

        try {
            if ($this->editMode) {
                $resource = Resource::findOrFail($this->resourceId);
                
                // Handle file upload if a new file is provided
                if ($this->file) {
                    // Delete old file
                    if ($resource->file_path && Storage::disk('public')->exists($resource->file_path)) {
                        Storage::disk('public')->delete($resource->file_path);
                    }
                    
                    // Store new file
                    $filePath = $this->file->store('resources', 'public');
                    $resourceData['file_path'] = $filePath;
                    $resourceData['file_name'] = $this->file->getClientOriginalName();
                    $resourceData['file_type'] = $this->file->getClientOriginalExtension();
                    $resourceData['file_size'] = $this->file->getSize();
                }
                
                $resource->update($resourceData);
                
                FlashMessageService::success('Resource updated successfully!', 'Resource Updated');
                $this->dispatch('flashMessage', 'success', 'Resource updated successfully!', 'Resource Updated', 5000);
            } else {
                // Store file
                $filePath = $this->file->store('resources', 'public');
                $resourceData['file_path'] = $filePath;
                $resourceData['file_name'] = $this->file->getClientOriginalName();
                $resourceData['file_type'] = $this->file->getClientOriginalExtension();
                $resourceData['file_size'] = $this->file->getSize();
                
                $resource = Resource::create($resourceData);
                
                FlashMessageService::success('Resource created successfully!', 'Resource Created');
                $this->dispatch('flashMessage', 'success', 'Resource created successfully!', 'Resource Created', 5000);
            }

            return redirect()->route('admin.resources');
            
        } catch (\Exception $e) {
            if ($this->editMode) {
                FlashMessageService::error('Failed to update resource. Please try again.', 'Update Failed');
                $this->dispatch('flashMessage', 'error', 'Failed to update resource. Please try again.', 'Update Failed', 8000);
            } else {
                FlashMessageService::error('Failed to create resource. Please try again.', 'Creation Failed');
                $this->dispatch('flashMessage', 'error', 'Failed to create resource. Please try again.', 'Creation Failed', 8000);
            }
            
            \Log::error('Resource form save error: ' . $e->getMessage());
        }
    }

    public function cancel()
    {
        return redirect()->route('admin.resources');
    }

    public function render()
    {
        return view('livewire.admin.resource-form');
    }
}