<?php

namespace App\Livewire\Admin;

use App\Services\FlashMessageService;
use App\Models\Setting;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;

class Settings extends Component
{
    use WithFileUploads;

    public $settings = [];
    public $logo;
    public $logoPreview;

    protected $rules = [
        'logo' => 'nullable|image|max:2048',
    ];

    public function mount()
    {
        // Check permissions
        if (!auth()->user()->can('settings.view')) {
            abort(403, 'Unauthorized access to settings.');
        }
        
        // Load settings from database or config
        $this->loadSettings();
    }

    public function loadSettings()
    {
        // Load settings from database
        $this->settings = [
            // General Settings
            'site_name' => Setting::get('site_name', 'LLN Project'),
            'site_email' => Setting::get('site_email', 'admin@llnproject.com'),
            'site_phone' => Setting::get('site_phone', ''),
            'site_address' => Setting::get('site_address', ''),
            
            // Organisation Information
            'company_name' => Setting::get('company_name', 'Skills Australia Institute'),
            'learner_correspondence_name' => Setting::get('learner_correspondence_name', 'Skills Australia Institute'),
            'learner_correspondence_phone' => Setting::get('learner_correspondence_phone', '+61 410001429'),
            'learner_correspondence_email' => Setting::get('learner_correspondence_email', 'admissions@skillsaustralia.edu.au'),
            'learner_redirection_url' => Setting::get('learner_redirection_url', 'http://www.skillsaustralia.edu.au'),
            'admin_correspondence_name' => Setting::get('admin_correspondence_name', 'Skills Australia Institute'),
            'admin_correspondence_email' => Setting::get('admin_correspondence_email', 'admissions@skillsaustralia.edu.au'),
            
            // Quiz Settings
            'daily_summary_quiz_completion_email' => Setting::get('daily_summary_quiz_completion_email', true),
            'daily_summary_email_contact' => Setting::get('daily_summary_email_contact', 'default admin correspondence'),
            'immediate_quiz_completion_emails' => Setting::get('immediate_quiz_completion_emails', true),
            'immediate_completion_email_contact' => Setting::get('immediate_completion_email_contact', 'default admin correspondence'),
            'deleted_learner_notifications' => Setting::get('deleted_learner_notifications', false),
            'incomplete_reminders_to_learners' => Setting::get('incomplete_reminders_to_learners', false),
            'learner_pre_enrolment_report' => Setting::get('learner_pre_enrolment_report', false),
            'quiz_reminder_days' => Setting::get('quiz_reminder_days', 7),
            'quiz_reminder_gap_days' => Setting::get('quiz_reminder_gap_days', 3),
            'copyright_text' => Setting::get('copyright_text', '© 2024 LLN Project. All rights reserved.'),
            'site_logo' => Setting::get('site_logo', ''),
            
            // System Settings
            'maintenance_mode' => Setting::get('maintenance_mode', false),
            'user_registration' => Setting::get('user_registration', true),
            'email_notifications' => Setting::get('email_notifications', true),
            'session_timeout' => Setting::get('session_timeout', 120),
            'max_login_attempts' => Setting::get('max_login_attempts', 5),
            'password_min_length' => Setting::get('password_min_length', 8),
            'require_password_confirmation' => Setting::get('require_password_confirmation', true),
        ];
    }

    public function updateSettings()
    {
        // Check permissions
        if (!auth()->user()->can('settings.edit')) {
            abort(403, 'Unauthorized to edit settings.');
        }
        
        $this->validate([
            // General Settings
            'settings.site_name' => 'required|string|max:255',
            'settings.site_email' => 'required|email',
            'settings.site_phone' => 'nullable|string|max:20',
            'settings.site_address' => 'nullable|string|max:500',
            
            // Organisation Information
            'settings.company_name' => 'required|string|max:255',
            'settings.learner_correspondence_name' => 'required|string|max:255',
            'settings.learner_correspondence_phone' => 'required|string|max:20',
            'settings.learner_correspondence_email' => 'required|email',
            'settings.learner_redirection_url' => 'required|url',
            'settings.admin_correspondence_name' => 'required|string|max:255',
            'settings.admin_correspondence_email' => 'required|email',
            
            // Quiz Settings
            'settings.daily_summary_email_contact' => 'nullable|string|max:255',
            'settings.immediate_completion_email_contact' => 'nullable|string|max:255',
            'settings.quiz_reminder_days' => 'required|integer|min:1|max:30',
            'settings.quiz_reminder_gap_days' => 'required|integer|min:1|max:14',
            'settings.copyright_text' => 'required|string|max:500',
            'logo' => 'nullable|image|max:2048',
            
            // System Settings
            'settings.session_timeout' => 'required|integer|min:30|max:480',
            'settings.max_login_attempts' => 'required|integer|min:3|max:10',
            'settings.password_min_length' => 'required|integer|min:6|max:20',
        ]);

        try {
            // Save general settings to database
            Setting::set('site_name', $this->settings['site_name'], 'string', 'Site name');
            Setting::set('site_email', $this->settings['site_email'], 'string', 'Site email address');
            Setting::set('site_phone', $this->settings['site_phone'], 'string', 'Site phone number');
            Setting::set('site_address', $this->settings['site_address'], 'string', 'Site address');
            
            // Save organisation information to database
            Setting::set('company_name', $this->settings['company_name'], 'string', 'Company name');
            Setting::set('learner_correspondence_name', $this->settings['learner_correspondence_name'], 'string', 'Learner correspondence name');
            Setting::set('learner_correspondence_phone', $this->settings['learner_correspondence_phone'], 'string', 'Learner correspondence phone');
            Setting::set('learner_correspondence_email', $this->settings['learner_correspondence_email'], 'string', 'Learner correspondence email');
            Setting::set('learner_redirection_url', $this->settings['learner_redirection_url'], 'string', 'Learner redirection URL');
            Setting::set('admin_correspondence_name', $this->settings['admin_correspondence_name'], 'string', 'Admin correspondence name');
            Setting::set('admin_correspondence_email', $this->settings['admin_correspondence_email'], 'string', 'Admin correspondence email');
            
            // Save quiz settings to database
            Setting::set('daily_summary_quiz_completion_email', $this->settings['daily_summary_quiz_completion_email'], 'boolean', 'Daily summary quiz completion email');
            Setting::set('daily_summary_email_contact', $this->settings['daily_summary_email_contact'], 'string', 'Daily summary email contact');
            Setting::set('immediate_quiz_completion_emails', $this->settings['immediate_quiz_completion_emails'], 'boolean', 'Immediate quiz completion emails');
            Setting::set('immediate_completion_email_contact', $this->settings['immediate_completion_email_contact'], 'string', 'Immediate completion email contact');
            Setting::set('deleted_learner_notifications', $this->settings['deleted_learner_notifications'], 'boolean', 'Deleted learner notifications');
            Setting::set('incomplete_reminders_to_learners', $this->settings['incomplete_reminders_to_learners'], 'boolean', 'Incomplete reminders to learners');
            Setting::set('learner_pre_enrolment_report', $this->settings['learner_pre_enrolment_report'], 'boolean', 'Learner pre-enrolment report');
            Setting::set('quiz_reminder_days', $this->settings['quiz_reminder_days'], 'integer', 'Days to send quiz reminders');
            Setting::set('quiz_reminder_gap_days', $this->settings['quiz_reminder_gap_days'], 'integer', 'Gap between quiz reminders in days');
            Setting::set('copyright_text', $this->settings['copyright_text'], 'string', 'Copyright text');
            
            // Handle logo upload
            if ($this->logo) {
                // Delete old logo if exists
                if ($this->settings['site_logo']) {
                    Storage::disk('public')->delete($this->settings['site_logo']);
                }
                
                // Store new logo
                $logoPath = $this->logo->store('logos', 'public');
                Setting::set('site_logo', $logoPath, 'string', 'Site logo');
                $this->settings['site_logo'] = $logoPath;
                
                // Clear cache to ensure logo updates immediately
                cache()->forget('settings');
            }
            
            // Save system settings to database
            Setting::set('maintenance_mode', $this->settings['maintenance_mode'], 'boolean', 'Maintenance mode');
            Setting::set('user_registration', $this->settings['user_registration'], 'boolean', 'Allow user registration');
            Setting::set('email_notifications', $this->settings['email_notifications'], 'boolean', 'Email notifications');
            Setting::set('session_timeout', $this->settings['session_timeout'], 'integer', 'Session timeout in minutes');
            Setting::set('max_login_attempts', $this->settings['max_login_attempts'], 'integer', 'Maximum login attempts');
            Setting::set('password_min_length', $this->settings['password_min_length'], 'integer', 'Minimum password length');
            Setting::set('require_password_confirmation', $this->settings['require_password_confirmation'], 'boolean', 'Require password confirmation');
            
            // Clear all settings cache
            cache()->forget('settings');
            
            FlashMessageService::success('Settings updated successfully!', 'Settings Updated');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'success', 'Settings updated successfully!', 'Settings Updated', 5000);
            
            // If logo was uploaded, refresh the page to update sidebar
            if ($this->logo) {
                $this->dispatch('logo-updated');
            }
            
        } catch (\Exception $e) {
            FlashMessageService::error('Failed to update settings. Please try again.', 'Update Failed');
            
            // Dispatch flash message event
            $this->dispatch('flashMessage', 'error', 'Failed to update settings. Please try again.', 'Update Failed', 8000);
            
            \Log::error('Settings update error: ' . $e->getMessage());
        }
    }

    public function updatedLogo()
    {
        $this->validateOnly('logo');
        $this->logoPreview = $this->logo->temporaryUrl();
    }

    public function removeLogo()
    {
        if ($this->settings['site_logo']) {
            Storage::disk('public')->delete($this->settings['site_logo']);
            Setting::set('site_logo', '', 'string', 'Site logo');
            $this->settings['site_logo'] = '';
            
            // Clear cache to ensure logo updates immediately
            cache()->forget('settings');
        }
        $this->logo = null;
        $this->logoPreview = null;
        
        FlashMessageService::success('Logo removed successfully!', 'Logo Removed');
        $this->dispatch('flashMessage', 'success', 'Logo removed successfully!', 'Logo Removed', 3000);
    }


    public function render()
    {
        return view('livewire.admin.settings');
    }
}
