<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class QuizAttemptAnswer extends Model
{
    use HasFactory;

    protected $fillable = [
        'attempt_id',
        'question_id',
        'answer_data',
        'scoring_data',
        'score',
        'max_score',
        'is_correct',
        'is_partial',
        'feedback',
        'manual_feedback',
        'time_spent',
        'attempts_count',
        'answered_at',
    ];

    protected $casts = [
        'answer_data' => 'array',
        'scoring_data' => 'array',
        'feedback' => 'array',
        'score' => 'decimal:2',
        'max_score' => 'decimal:2',
        'is_correct' => 'boolean',
        'is_partial' => 'boolean',
        'answered_at' => 'datetime',
    ];

    // Relationships
    public function attempt(): BelongsTo
    {
        return $this->belongsTo(QuizAttempt::class);
    }

    public function question(): BelongsTo
    {
        return $this->belongsTo(QuizQuestion::class);
    }

    // Accessors
    public function getAnswerTextAttribute(): string
    {
        if (is_array($this->answer_data)) {
            return implode(', ', $this->answer_data);
        }

        return (string) $this->answer_data;
    }

    public function getScorePercentageAttribute(): float
    {
        if ($this->max_score == 0) {
            return 0;
        }

        return round(($this->score / $this->max_score) * 100, 2);
    }

    public function getIsFullyCorrectAttribute(): bool
    {
        return $this->is_correct && !$this->is_partial;
    }

    public function getIsPartiallyCorrectAttribute(): bool
    {
        return $this->is_partial;
    }

    public function getIsIncorrectAttribute(): bool
    {
        return !$this->is_correct && !$this->is_partial;
    }

    // Methods
    public function getAnswerData(string $key = null, $default = null)
    {
        if ($key === null) {
            return $this->answer_data ?? [];
        }

        return data_get($this->answer_data, $key, $default);
    }

    public function setAnswerData(array $data): void
    {
        $this->update(['answer_data' => array_merge($this->answer_data ?? [], $data)]);
    }

    public function getScoringData(string $key = null, $default = null)
    {
        if ($key === null) {
            return $this->scoring_data ?? [];
        }

        return data_get($this->scoring_data, $key, $default);
    }

    public function setScoringData(array $data): void
    {
        $this->update(['scoring_data' => array_merge($this->scoring_data ?? [], $data)]);
    }

    public function getFeedback(string $key = null, $default = null)
    {
        if ($key === null) {
            return $this->feedback ?? [];
        }

        return data_get($this->feedback, $key, $default);
    }

    public function setFeedback(array $feedback): void
    {
        $this->update(['feedback' => array_merge($this->feedback ?? [], $feedback)]);
    }

    public function addManualFeedback(string $feedback, int $userId): void
    {
        $this->update([
            'manual_feedback' => $feedback,
            'updated_by' => $userId,
        ]);
    }

    public function updateScore(float $score, array $scoringData = []): void
    {
        $this->update([
            'score' => $score,
            'scoring_data' => array_merge($this->scoring_data ?? [], $scoringData),
            'is_correct' => $score >= $this->max_score,
            'is_partial' => $score > 0 && $score < $this->max_score,
        ]);
    }

    public function incrementAttempts(): void
    {
        $this->increment('attempts_count');
    }

    public function getFormattedAnswer(): string
    {
        $question = $this->question;
        
        if (!$question) {
            return $this->answer_text;
        }

        switch ($question->type) {
            case 'MULTIPLE_CHOICE':
                return $this->formatMultipleChoiceAnswer();
            case 'TRUE_FALSE':
                return $this->formatTrueFalseAnswer();
            case 'FILL_BLANK':
                return $this->formatFillBlankAnswer();
            case 'SHORT_ANSWER':
            case 'ESSAY':
                return $this->formatTextAnswer();
            case 'MATCH':
                return $this->formatMatchAnswer();
            case 'ORDER':
                return $this->formatOrderAnswer();
            case 'MATRIX':
                return $this->formatMatrixAnswer();
            case 'FILE_UPLOAD':
                return $this->formatFileUploadAnswer();
            case 'DATE':
                return $this->formatDateAnswer();
            case 'CODE':
                return $this->formatCodeAnswer();
            case 'NUMERIC':
                return $this->formatNumericAnswer();
            case 'RATING':
            case 'LIKERT':
                return $this->formatRatingAnswer();
            default:
                return $this->answer_text;
        }
    }

    private function formatMultipleChoiceAnswer(): string
    {
        $answer = $this->answer_data;
        $question = $this->question;
        $options = $question->getAnswerData('options', []);
        
        if (is_array($answer)) {
            $selectedOptions = [];
            foreach ($answer as $selectedValue) {
                foreach ($options as $option) {
                    if ($option['value'] === $selectedValue) {
                        $selectedOptions[] = $option['label'] ?? $option['value'];
                        break;
                    }
                }
            }
            return implode(', ', $selectedOptions);
        }
        
        return $answer;
    }

    private function formatTrueFalseAnswer(): string
    {
        return $this->answer_data ? 'True' : 'False';
    }

    private function formatFillBlankAnswer(): string
    {
        if (is_array($this->answer_data)) {
            return implode(' | ', $this->answer_data);
        }
        
        return $this->answer_data;
    }

    private function formatTextAnswer(): string
    {
        return $this->answer_data;
    }

    private function formatMatchAnswer(): string
    {
        if (!is_array($this->answer_data)) {
            return $this->answer_data;
        }

        $formatted = [];
        foreach ($this->answer_data as $match) {
            if (isset($match['left']) && isset($match['right'])) {
                $formatted[] = "{$match['left']} → {$match['right']}";
            }
        }
        
        return implode('; ', $formatted);
    }

    private function formatOrderAnswer(): string
    {
        if (!is_array($this->answer_data)) {
            return $this->answer_data;
        }

        return implode(' → ', $this->answer_data);
    }

    private function formatMatrixAnswer(): string
    {
        if (!is_array($this->answer_data)) {
            return $this->answer_data;
        }

        $formatted = [];
        foreach ($this->answer_data as $row) {
            if (is_array($row)) {
                $formatted[] = implode(' | ', $row);
            } else {
                $formatted[] = $row;
            }
        }
        
        return implode('; ', $formatted);
    }

    private function formatFileUploadAnswer(): string
    {
        if (!is_array($this->answer_data)) {
            return $this->answer_data;
        }

        $files = [];
        foreach ($this->answer_data as $file) {
            if (isset($file['name'])) {
                $files[] = $file['name'];
            }
        }
        
        return implode(', ', $files);
    }

    private function formatDateAnswer(): string
    {
        try {
            return \Carbon\Carbon::parse($this->answer_data)->format('Y-m-d');
        } catch (\Exception $e) {
            return $this->answer_data;
        }
    }

    private function formatCodeAnswer(): string
    {
        return $this->answer_data;
    }

    private function formatNumericAnswer(): string
    {
        return (string) $this->answer_data;
    }

    private function formatRatingAnswer(): string
    {
        return $this->answer_data . '/5';
    }
}