<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class EmailNotificationService
{
    /**
     * Check if email notifications are enabled
     */
    public static function isEmailNotificationsEnabled(): bool
    {
        return Setting::get('email_notifications', true);
    }

    /**
     * Send email only if notifications are enabled
     */
    public static function sendEmailIfEnabled($mailable, $to, $subject = null): bool
    {
        if (!self::isEmailNotificationsEnabled()) {
            Log::info('Email notifications are disabled. Email not sent.', [
                'to' => $to,
                'subject' => $subject,
                'mailable' => get_class($mailable)
            ]);
            return false;
        }

        try {
            Mail::to($to)->send($mailable);
            Log::info('Email sent successfully', [
                'to' => $to,
                'subject' => $subject,
                'mailable' => get_class($mailable)
            ]);
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send email', [
                'to' => $to,
                'subject' => $subject,
                'mailable' => get_class($mailable),
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send daily summary quiz completion email if enabled
     */
    public static function sendDailySummaryQuizEmail($to, $data): bool
    {
        if (!Setting::get('daily_summary_quiz_completion_email', true)) {
            Log::info('Daily summary quiz completion email is disabled');
            return false;
        }

        if (!self::isEmailNotificationsEnabled()) {
            Log::info('Email notifications are disabled. Daily summary quiz email not sent.');
            return false;
        }

        try {
            // You can create a specific mailable for this
            // Mail::to($to)->send(new DailySummaryQuizMail($data));
            Log::info('Daily summary quiz email sent', ['to' => $to]);
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send daily summary quiz email', [
                'to' => $to,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send immediate quiz completion email if enabled
     */
    public static function sendImmediateQuizCompletionEmail($to, $data): bool
    {
        if (!Setting::get('immediate_quiz_completion_emails', true)) {
            Log::info('Immediate quiz completion emails are disabled');
            return false;
        }

        if (!self::isEmailNotificationsEnabled()) {
            Log::info('Email notifications are disabled. Immediate quiz completion email not sent.');
            return false;
        }

        try {
            // You can create a specific mailable for this
            // Mail::to($to)->send(new ImmediateQuizCompletionMail($data));
            Log::info('Immediate quiz completion email sent', ['to' => $to]);
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send immediate quiz completion email', [
                'to' => $to,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send deleted learner notification if enabled
     */
    public static function sendDeletedLearnerNotification($to, $learnerData): bool
    {
        if (!Setting::get('deleted_learner_notifications', false)) {
            Log::info('Deleted learner notifications are disabled');
            return false;
        }

        if (!self::isEmailNotificationsEnabled()) {
            Log::info('Email notifications are disabled. Deleted learner notification not sent.');
            return false;
        }

        try {
            // You can create a specific mailable for this
            // Mail::to($to)->send(new DeletedLearnerNotificationMail($learnerData));
            Log::info('Deleted learner notification sent', ['to' => $to]);
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send deleted learner notification', [
                'to' => $to,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send incomplete reminders to learners if enabled
     */
    public static function sendIncompleteReminders($to, $data): bool
    {
        if (!Setting::get('incomplete_reminders_to_learners', false)) {
            Log::info('Incomplete reminders to learners are disabled');
            return false;
        }

        if (!self::isEmailNotificationsEnabled()) {
            Log::info('Email notifications are disabled. Incomplete reminder not sent.');
            return false;
        }

        try {
            // You can create a specific mailable for this
            // Mail::to($to)->send(new IncompleteReminderMail($data));
            Log::info('Incomplete reminder sent', ['to' => $to]);
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send incomplete reminder', [
                'to' => $to,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send learner pre-enrolment report if enabled
     */
    public static function sendLearnerPreEnrolmentReport($to, $data): bool
    {
        if (!Setting::get('learner_pre_enrolment_report', false)) {
            Log::info('Learner pre-enrolment report is disabled');
            return false;
        }

        if (!self::isEmailNotificationsEnabled()) {
            Log::info('Email notifications are disabled. Pre-enrolment report not sent.');
            return false;
        }

        try {
            // You can create a specific mailable for this
            // Mail::to($to)->send(new LearnerPreEnrolmentReportMail($data));
            Log::info('Learner pre-enrolment report sent', ['to' => $to]);
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send learner pre-enrolment report', [
                'to' => $to,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}

