<?php

namespace App\Services;

use App\Models\EmailTemplate;
use App\Models\User;
use App\Models\Setting;
use Illuminate\Support\Facades\Mail;

class EmailTemplateService
{
    /**
     * Get a template by slug
     */
    public static function getTemplateBySlug($slug)
    {
        return EmailTemplate::with('templateType')
            ->where('slug', $slug)
            ->where('is_active', true)
            ->first();
    }

    /**
     * Process and send email using template
     */
    public static function sendTemplateEmail($templateSlug, $recipientEmail, $data = [])
    {
        $template = self::getTemplateBySlug($templateSlug);
        
        if (!$template) {
            \Log::warning("Email template '{$templateSlug}' not found or inactive");
            throw new \Exception("Email template '{$templateSlug}' not found or inactive");
        }

        // Check if template type is active
        if ($template->templateType && !$template->templateType->is_active) {
            \Log::warning("Email template type '{$template->templateType->name}' is inactive. Email sending blocked for template '{$templateSlug}'");
            throw new \Exception("Email template type '{$template->templateType->name}' is inactive. Email sending is blocked.");
        }

        // Merge with default system data
        $data = array_merge(self::getDefaultData(), $data);

        // Process template
        $processedTemplate = $template->processTemplate($data);

        // Send email
        Mail::raw($processedTemplate['body'], function ($message) use ($processedTemplate, $recipientEmail) {
            $message->to($recipientEmail)
                   ->subject($processedTemplate['subject']);
        });

        \Log::info("Email sent successfully using template '{$templateSlug}' to '{$recipientEmail}'");
        return true;
    }

    /**
     * Get processed template content
     */
    public static function getProcessedTemplate($templateSlug, $data = [])
    {
        $template = self::getTemplateBySlug($templateSlug);
        
        if (!$template) {
            \Log::warning("Email template '{$templateSlug}' not found or inactive");
            throw new \Exception("Email template '{$templateSlug}' not found or inactive");
        }

        // Check if template type is active
        if ($template->templateType && !$template->templateType->is_active) {
            \Log::warning("Email template type '{$template->templateType->name}' is inactive. Template processing blocked for template '{$templateSlug}'");
            throw new \Exception("Email template type '{$template->templateType->name}' is inactive. Template processing is blocked.");
        }

        // Merge with default system data
        $data = array_merge(self::getDefaultData(), $data);

        return $template->processTemplate($data);
    }

    /**
     * Get default system data for templates
     */
    public static function getDefaultData()
    {
        $siteLogo = Setting::get('site_logo', '');
        $logoUrl = '';
        
        if ($siteLogo && \Storage::disk('public')->exists($siteLogo)) {
            $logoUrl = asset('storage/' . $siteLogo);
        }
        
        return [
            'site_name' => Setting::get('site_name', 'LLN Project'),
            'site_url' => config('app.url'),
            'company_name' => Setting::get('site_name', 'LLN Project'),
            'company_email' => Setting::get('site_email', config('mail.from.address')),
            'company_phone' => Setting::get('site_phone', ''),
            'company_address' => Setting::get('site_address', ''),
            'current_date' => now()->format('Y-m-d'),
            'current_time' => now()->format('H:i:s'),
            'year' => now()->year,
            'company_logo' => $logoUrl,
            'site_logo' => $logoUrl,
            'logo_url' => $logoUrl,
        ];
    }

    /**
     * Get data for a specific user
     */
    public static function getUserData(User $user)
    {
        return [
            'learner_first_name' => $user->name ? explode(' ', $user->name)[0] : '',
            'learner_last_name' => $user->name ? implode(' ', array_slice(explode(' ', $user->name), 1)) : '',
            'learner_full_name' => $user->name,
            'learner_email' => $user->email,
            'learner_phone' => $user->phone ?? '',
        ];
    }

    /**
     * Get data for admin user
     */
    public static function getAdminData(User $admin)
    {
        return [
            'admin_first_name' => $admin->name ? explode(' ', $admin->name)[0] : '',
            'admin_last_name' => $admin->name ? implode(' ', array_slice(explode(' ', $admin->name), 1)) : '',
            'admin_full_name' => $admin->name,
            'admin_email' => $admin->email,
        ];
    }

    /**
     * Check if template type is active
     */
    public static function isTemplateTypeActive($templateSlug)
    {
        $template = self::getTemplateBySlug($templateSlug);
        
        if (!$template) {
            return false;
        }

        if (!$template->templateType) {
            return true; // If no template type is assigned, allow processing
        }

        return $template->templateType->is_active;
    }

    /**
     * Get template type status
     */
    public static function getTemplateTypeStatus($templateSlug)
    {
        $template = self::getTemplateBySlug($templateSlug);
        
        if (!$template) {
            return [
                'active' => false,
                'message' => 'Template not found or inactive'
            ];
        }

        if (!$template->templateType) {
            return [
                'active' => true,
                'message' => 'No template type assigned'
            ];
        }

        return [
            'active' => $template->templateType->is_active,
            'message' => $template->templateType->is_active 
                ? 'Template type is active' 
                : 'Template type is inactive',
            'type_name' => $template->templateType->name
        ];
    }

    /**
     * Validate template variables
     */
    public static function validateTemplateVariables($template, $data)
    {
        $missingVariables = [];
        
        // Extract variables from template content
        preg_match_all('/\{([^}]+)\}/', $template->subject . ' ' . $template->body, $matches);
        $requiredVariables = array_unique($matches[1]);
        
        foreach ($requiredVariables as $variable) {
            if (!isset($data[$variable]) || empty($data[$variable])) {
                $missingVariables[] = $variable;
            }
        }
        
        return $missingVariables;
    }
}
