<?php

namespace App\Services;

class FlashMessageService
{
    /**
     * Show a success flash message
     */
    public static function success(string $message, ?string $title = null, int $duration = 5000): void
    {
        self::setFlashMessage('success', $message, $title, $duration);
    }

    /**
     * Show an error flash message
     */
    public static function error(string $message, ?string $title = null, int $duration = 8000): void
    {
        self::setFlashMessage('error', $message, $title, $duration);
    }

    /**
     * Show a warning flash message
     */
    public static function warning(string $message, ?string $title = null, int $duration = 6000): void
    {
        self::setFlashMessage('warning', $message, $title, $duration);
    }

    /**
     * Show an info flash message
     */
    public static function info(string $message, ?string $title = null, int $duration = 5000): void
    {
        self::setFlashMessage('info', $message, $title, $duration);
    }

    /**
     * Set a flash message in session
     */
    private static function setFlashMessage(string $type, string $message, ?string $title = null, int $duration = 5000): void
    {
        session()->flash($type, [
            'message' => $message,
            'title' => $title,
            'duration' => $duration,
            'timestamp' => now()
        ]);
    }

    /**
     * Get flash message data
     */
    public static function getFlashMessage(string $type): ?array
    {
        return session($type);
    }

    /**
     * Clear all flash messages
     */
    public static function clear(): void
    {
        session()->forget(['success', 'error', 'warning', 'info']);
    }

    /**
     * User Management specific messages
     */
    public static function userCreated(): void
    {
        self::success('User created successfully!', 'Success');
    }

    public static function userUpdated(): void
    {
        self::success('User updated successfully!', 'Success');
    }

    public static function userDeleted(): void
    {
        self::success('User deleted successfully!', 'Success');
    }

    public static function userCreateFailed(): void
    {
        self::error('Failed to create user. Please try again.', 'Error');
    }

    public static function userUpdateFailed(): void
    {
        self::error('Failed to update user. Please try again.', 'Error');
    }

    public static function userDeleteFailed(): void
    {
        self::error('Failed to delete user. Please try again.', 'Error');
    }

    public static function cannotDeleteSelf(): void
    {
        self::error('You cannot delete your own account!', 'Warning');
    }

    public static function validationFailed(): void
    {
        self::error('Please fix the validation errors and try again.', 'Validation Error');
    }

    /**
     * Role-specific messages
     */
    public static function insufficientPermissions(): void
    {
        self::error('You do not have sufficient permissions to perform this action.', 'Access Denied');
    }

    public static function roleAssigned(string $role): void
    {
        self::success("Role '{$role}' has been assigned successfully.", 'Role Assigned');
    }

    public static function roleRemoved(string $role): void
    {
        self::success("Role '{$role}' has been removed successfully.", 'Role Removed');
    }

    public static function roleAssignmentFailed(): void
    {
        self::error('Failed to assign role. Please try again.', 'Role Assignment Error');
    }

    /**
     * Permission Management Messages
     */
    public static function permissionGranted(string $permission, string $role): void
    {
        self::success("Permission '{$permission}' has been granted to role '{$role}'.", 'Permission Granted');
    }

    public static function permissionRevoked(string $permission, string $role): void
    {
        self::success("Permission '{$permission}' has been revoked from role '{$role}'.", 'Permission Revoked');
    }

    public static function permissionAssignmentFailed(): void
    {
        self::error('Failed to update permissions. Please try again.', 'Permission Error');
    }

    /**
     * LLN Specialist specific messages
     */
    public static function specialistActionCompleted(): void
    {
        self::success('Specialist action completed successfully!', 'Success');
    }

    /**
     * Enrolment Officer specific messages
     */
    public static function enrolmentCompleted(): void
    {
        self::success('Enrolment process completed successfully!', 'Enrolment Success');
    }

    public static function enrolmentFailed(): void
    {
        self::error('Enrolment process failed. Please try again.', 'Enrolment Error');
    }

    /**
     * Student specific messages
     */
    public static function studentActionCompleted(): void
    {
        self::success('Student action completed successfully!', 'Success');
    }

    /**
     * Generic operation messages
     */
    public static function operationSuccess(string $operation): void
    {
        self::success("{$operation} completed successfully!", 'Success');
    }

    public static function operationFailed(string $operation): void
    {
        self::error("Failed to {$operation}. Please try again.", 'Error');
    }

    /**
     * Form validation messages
     */
    public static function formValidationError(): void
    {
        self::error('Please check the form for errors and try again.', 'Form Validation Error');
    }

    public static function requiredFieldsMissing(): void
    {
        self::warning('Please fill in all required fields.', 'Required Fields');
    }
}
